/******************************************************************************
 *                    OpenGL Demonstration                                    *
 *                    ====================                                    *
 *                                                                            *
 * Author: Martin Christian                                                   *
 * Licence: Public Domain                                                     *
 *                                                                            *
 * Credits:                                                                   *
 *   -> OpenGL lecturer: Jens-Peer Kuska <kuska@izbi.uni-leipzig.de>          *
 *   -> Lighthouse Tutorials: http://www.lighthouse3d.com/opengl/             *
 *   -> GLUT: http://www.opengl.org/resources/libraries/glut/spec3/spec3.html *
 *   -> SIGGRAPH '98: http://www.sgi.com/software/opengl/advanced98/notes/    *
 ******************************************************************************/


/**************************** INCLUDES ****************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>
#include "texture.h"

/**************************** CONSTANTS ***************************************/
#ifndef PI
  #define PI 3.14159265358979323846
#endif
#define RAD_FACTOR 0.01745329251994329576

/**************************** DATA TYPES **************************************/
typedef struct Point3D {
	double x, y, z;
} Point3D;

typedef struct RGBAColor {
	double r, g, b, a;
} RGBAColor;

/************************** DATA STRUCTURES ***********************************/
int windowHandle = -1;
GLboolean animate = GL_TRUE;

/* Menu */
#define MENU_QUIT 0
#define MENU_DAYLIGHT 1
#define MENU_SUNSET 2

/* Display Lists */
#define N_LISTS 4
GLuint marble_pillar_list;
GLuint stone_pillar_list;
GLuint quad_list;
GLuint ground_list;

/* Light */
float light_position[4] = { -1.0, 4.0, 1.0, 0.0 };
float light_ambient[4] = { 0.2, 0.2, 0.2, 1.0 };
float light_daylight[4] = { 1.0, 1.0, 1.0, 1.0 };
float light_sunset[4] = { 0.39, 0.13, 0.0, 1.0 };
float light_specular[4] = { 1.0, 1.0, 1.0, 1.0 };

/* Material */
GLfloat mat_marble_specular[4] = { 1.0, 1.0, 1.0, 1.0 };
GLfloat mat_stone_specular[4] = { 0.8, 0.8, 0.8, 1.0 };
GLfloat mat_meadow_specular[4] = { 0.0, 0.0, 0.0, 1.0 };
GLfloat mat_default_specular[4] = { 0.0, 0.0, 0.0, 1.0 };

GLfloat mat_marble_diffuse[4] = { 0.93, 0.81, 0.68, 1.0 };
GLfloat mat_stone_diffuse[4] = { 0.63, 0.66, 0.66, 1.0 };
GLfloat mat_meadow_diffuse[4] = { 0.45, 0.50, 0.17, 1.0 };
GLfloat mat_default_diffuse[4] = { 0.8, 0.8, 0.8, 1.0 };

GLfloat mat_marble_shine = 124.0;
GLfloat mat_stone_shine = 32.0;
GLfloat mat_meadow_shine = 0.0;
GLfloat mat_default_shine = 0.0;

/* Camera */
#define VIEWANGLE 45.0
#define NEARPLANE 1.0
#define FARPLANE 6.0
#define Y_SOCKET 1.0
#define CAMERA_RADIUS 3.0
#define ROTATION_STEP 1.0
Point3D camera_position = { 0.0, Y_SOCKET, CAMERA_RADIUS };
Point3D camera_tilting = { 0.0, 1.0, 0.0 };
GLboolean perspective = GL_TRUE;
GLdouble camera_hrotation = 0.0;        /* horizontal camera rotation */
GLdouble camera_vrotation = 0.0;        /* vertical camera rotation */
GLint viewport_width = 400;
GLint viewport_height = 400;
GLdouble viewport_ration = 1.0;         /* viewport_width : viewport_height */

/* Textures */
#define TEXTURE_COUNT 3
const char* textureFiles[TEXTURE_COUNT] = { "gras.png", "marmor.png", "stone.png" };
extern Texture *textureHandles;
extern int textureCount;

/******************************************************************************
 *                         drawPillar                                         *
 ******************************************************************************/
void drawPillar( GLfloat height, GLfloat radius, int nsides, GLboolean wire ) {
	int i;
	GLfloat alpha, x, z, tex_step, tex_start;

	alpha = 360.0 / nsides;
	x = cos( alpha * RAD_FACTOR ) * radius;
	z = sin( alpha * RAD_FACTOR ) * radius;
	tex_step = 1.0 / nsides;
	tex_start = 0.0;

	glPushMatrix();
	for ( i=0; i<nsides; i++ ) {
		glBegin( wire ? GL_LINE_STRIP : GL_TRIANGLE_STRIP );
			glTexCoord2d( 0.0, 0.0 );
			glNormal3f( 0.0, -1.0, 0.0 );
			glVertex3f( 0.0, 0.0, 0.0 ); /* 1 */

			glTexCoord2d( tex_start+tex_step, 1.0 );
			glNormal3f( 1.0, -1.0, 0.0 );
			glVertex3f( radius, 0.0, 0.0 ); /* 2 */

			glTexCoord2d( tex_start, 1.0 );
			glNormal3f( x, -1.0, z );
			glVertex3f( x, 0.0, z ); /* 3 */

			glTexCoord2d( tex_start+tex_step, 0.0 );
			glNormal3f( 1.0, 1.0, 0.0 );
			glVertex3f( radius, height, 0.0 ); /* 4 */

			glTexCoord2d( tex_start, 0.0 );
			glNormal3f( x, 1.0, z );
			glVertex3f( x, height, z ); /* 5 */

			glTexCoord2d( 0.0, 0.0 );
			glNormal3f( 0.0, 1.0, 0.0 );
			glVertex3f( 0.0, height, 0.0 ); /* 6 */
		glEnd();
		glRotatef( alpha, 0.0, 1.0, 0.0 );
		tex_start += tex_step;
	}
	glPopMatrix();
}

/******************************************************************************
 *                         drawQuad                                           *
 ******************************************************************************/
void drawQuad( GLfloat width, GLfloat height, GLfloat depth ) {
	GLfloat tex_step, tex_start;

	tex_step = 1.0 / ( 2.0*height + depth );
	tex_start = 0.0;

	/* from front to back */
	glBegin( GL_QUAD_STRIP );
		glTexCoord2d( 1.0, tex_start );
		glNormal3f( -1.0, -1.0, -1.0 );
		glVertex3f( 0.0, 0.0, 0.0 ); /* 1 */

		glTexCoord2d( 0.0, tex_start );
		glNormal3f( 1.0, -1.0, -1.0 );
		glVertex3f( width, 0.0, 0.0 ); /* 2 */

		tex_start += (height*tex_step);
		glTexCoord2d( 1.0, tex_start );
		glNormal3f( -1.0, 1.0, -1.0 );
		glVertex3f( 0.0, height, 0.0 ); /* 3 */

		glTexCoord2d( 0.0, tex_start );
		glNormal3f( 1.0, 1.0, -1.0 );
		glVertex3f( width, height, 0.0 ); /* 4 */

		tex_start += (depth*tex_step);
		glTexCoord2d( 1.0, tex_start );
		glNormal3f( -1.0, 1.0, 1.0 );
		glVertex3f( 0.0, height, depth ); /* 5 */

		glTexCoord2d( 0.0, tex_start );
		glNormal3f( 1.0, 1.0, 1.0 );
		glVertex3f( width, height, depth ); /* 6 */

		tex_start += (height*tex_step);
		glTexCoord2d( 1.0, tex_start );
		glNormal3f( -1.0, -1.0, 1.0 );
		glVertex3f( 0.0, 0.0, depth ); /* 7 */

		glTexCoord2d( 0.0, tex_start );
		glNormal3f( 1.0, -1.0, 1.0 );
		glVertex3f( width, 0.0, depth ); /* 8 */
	glEnd();

	/* from left to right */
	tex_start = 0.0;
	glBegin( GL_QUAD_STRIP );
		glTexCoord2d( 1.0, tex_start );
		glNormal3f( -1.0, 1.0, -1.0 );
		glVertex3f( 0.0, height, 0.0 ); /* 3 */

		glTexCoord2d( 0.0, tex_start );
		glNormal3f( -1.0, 1.0, 1.0 );
		glVertex3f( 0.0, height, depth ); /* 5 */

		tex_start += (height*tex_step);
		glTexCoord2d( 1.0, tex_start );
		glNormal3f( -1.0, -1.0, -1.0 );
		glVertex3f( 0.0, 0.0, 0.0 ); /* 1 */

		glTexCoord2d( 0.0, tex_start );
		glNormal3f( -1.0, -1.0, 1.0 );
		glVertex3f( 0.0, 0.0, depth ); /* 7 */

		tex_start += (width*tex_step);
		glTexCoord2d( 1.0, tex_start );
		glNormal3f( 1.0, -1.0, -1.0 );
		glVertex3f( width, 0.0, 0.0 ); /* 2 */

		glTexCoord2d( 0.0, tex_start );
		glNormal3f( 1.0, -1.0, 1.0 );
		glVertex3f( width, 0.0, depth ); /* 8 */

		tex_start += (height*tex_step);
		glTexCoord2d( 1.0, tex_start );
		glNormal3f( 1.0, 1.0, -1.0 );
		glVertex3f( width, height, 0.0 ); /* 4 */

		glTexCoord2d( 0.0, tex_start );
		glNormal3f( 1.0, 1.0, 1.0 );
		glVertex3f( width, height, depth ); /* 6 */
	glEnd();
}

/******************************************************************************
 *                         drawPlane                                          *
 ******************************************************************************/
void drawPlane( GLfloat width, GLfloat depth ) {
	glNormal3f( 0.0, 1.0, 0.0 );
	glBegin( GL_QUADS );
		glTexCoord2d( 0.0, 0.0 );
		glVertex3f( 0.0, 0.0, 0.0 );

		glTexCoord2d( 2.0, 0.0 );
		glVertex3f( 4.0, 0.0, 0.0 );

		glTexCoord2d( 2.0, 2.0 );
		glVertex3f( 4.0, 0.0, 4.0 );

		glTexCoord2d( 0.0, 2.0 );
		glVertex3f( 0.0, 0.0, 4.0 );
	glEnd();
}

/******************************************************************************
 *                         setCamera                                          *
 ******************************************************************************/
void setCamera() {
/*		glOrtho( -2.0, 2.0, -2.0, 2.0, NEARPLANE, FARPLANE );
      glOrtho( left, right, bottom, top, near_val, far_val ) */
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glViewport( 0, 0, viewport_width, viewport_height );
	gluPerspective( VIEWANGLE, viewport_ration, NEARPLANE, FARPLANE );
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	gluLookAt( camera_position.x, camera_position.y, camera_position.z,
	           0.0, 0.0, 0.0, /* look at origin */
	           camera_tilting.x, camera_tilting.y, camera_tilting.z );
}

/******************************************************************************
 *                         initScene                                          *
 ******************************************************************************/
void initScene( void ) {
	glClearColor( 0.0, 0.0, 0.0, 0.0 );
	glColor3f( 1.0, 1.0, 1.0 );

	glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );
	glEnable( GL_DEPTH_TEST );
	glEnable( GL_NORMALIZE );
	glShadeModel( GL_SMOOTH );

	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();
	glMatrixMode( GL_MODELVIEW );
	glLoadIdentity();

	/* light model */
	glEnable( GL_LIGHTING );
	glEnable( GL_LIGHT0 );
	glEnable( GL_COLOR_MATERIAL );
	glColorMaterial( GL_FRONT, GL_SPECULAR );
	glLightModelf( GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE );
	glLightfv( GL_LIGHT0, GL_AMBIENT, light_ambient );
	glLightfv( GL_LIGHT0, GL_DIFFUSE, light_daylight );
	glLightfv( GL_LIGHT0, GL_SPECULAR, light_specular );

	/* create display lists */
	marble_pillar_list = glGenLists( N_LISTS );
	stone_pillar_list = marble_pillar_list + 1;
	quad_list = marble_pillar_list + 2;
	ground_list = marble_pillar_list + 3;

	/* marble pillar */
	glNewList( marble_pillar_list, GL_COMPILE );
		glMaterialfv( GL_FRONT, GL_SPECULAR, mat_marble_specular );
		glMaterialfv( GL_FRONT, GL_DIFFUSE, mat_marble_diffuse );
		glMaterialf( GL_FRONT, GL_SHININESS, mat_marble_shine );
		glEnable( GL_TEXTURE_2D );
		glBindTexture( GL_TEXTURE_2D, textureHandles[1].texid );
		drawPillar( 0.8, 0.1, 15, GL_FALSE );
		glDisable( GL_TEXTURE_2D );
	glEndList();

	/* stone pillar */
	glNewList( stone_pillar_list, GL_COMPILE );
		glMaterialfv( GL_FRONT, GL_SPECULAR, mat_stone_specular );
		glMaterialfv( GL_FRONT, GL_DIFFUSE, mat_stone_diffuse );
		glMaterialf( GL_FRONT, GL_SHININESS, mat_default_shine );
		glEnable( GL_TEXTURE_2D );
		glBindTexture( GL_TEXTURE_2D, textureHandles[2].texid );
		drawPillar( 0.8, 0.1, 15, GL_FALSE );
		glDisable( GL_TEXTURE_2D );
	glEndList();

	/* marble quader */
	glNewList( quad_list, GL_COMPILE );
		glMaterialfv( GL_FRONT, GL_SPECULAR, mat_marble_specular );
		glMaterialfv( GL_FRONT, GL_DIFFUSE, mat_marble_diffuse );
		glMaterialf( GL_FRONT, GL_SHININESS, mat_marble_shine );
		glEnable( GL_TEXTURE_2D );
		glBindTexture( GL_TEXTURE_2D, textureHandles[1].texid );
		drawQuad( 1.2, 0.1, 1.2 );
		glDisable( GL_TEXTURE_2D );
	glEndList();

	/* meadow */
	glNewList( ground_list, GL_COMPILE );
		glMaterialfv( GL_FRONT_AND_BACK, GL_SPECULAR, mat_meadow_specular );
		glMaterialfv( GL_FRONT_AND_BACK, GL_DIFFUSE, mat_meadow_diffuse );
		glMaterialf( GL_FRONT_AND_BACK, GL_SHININESS, mat_meadow_shine );
		glEnable( GL_TEXTURE_2D );
		glBindTexture( GL_TEXTURE_2D, textureHandles[0].texid );
		drawPlane( 4.0, 4.0 );
		glDisable( GL_TEXTURE_2D );
	glEndList();
}

/******************************************************************************
 *                         resizeWindow                                       *
 ******************************************************************************/
void resizeWindow( int width, int height ) {
	if ( !height ) height = 1;
	viewport_ration = 1.0f * width / height;
	viewport_width = width;
	viewport_height = height;
	setCamera();
}

/******************************************************************************
 *                        animateScene                                        *
 ******************************************************************************/
void animateScene( void ) {
}

/******************************************************************************
 *                         renderScene                                        *
 ******************************************************************************/
void renderScene( void ) {
	glMatrixMode( GL_MODELVIEW );
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

	/* create ground */
	glPushMatrix();
	glTranslatef( -2.0, 0.0, -2.0 );
	glCallList( ground_list );
	glPopMatrix();

	/* create temple */
	glPushMatrix();
	glTranslatef( -0.6, -0.08, -0.6 );
	glCallList( quad_list );
	glPopMatrix();

	glPushMatrix();
	glTranslatef( 0.5, 0.02, 0.5 );
	glCallList( marble_pillar_list );
	glPopMatrix();

	glPushMatrix();
	glTranslatef( -0.5, 0.02, 0.5 );
	glCallList( marble_pillar_list );
	glPopMatrix();

	glPushMatrix();
	glTranslatef( -0.5, 0.02, -0.5 );
	glCallList( marble_pillar_list );
	glPopMatrix();

	glPushMatrix();
	glTranslatef( 0.5, 0.02, -0.5 );
	glCallList( marble_pillar_list );
	glPopMatrix();

	glPushMatrix();
	glTranslatef( -0.6, 0.8, -0.6 );
	glCallList( quad_list );
	glPopMatrix();

	/* create ruin */
	glPushMatrix();
	glTranslatef( -1.9, 0.0, -1.9 );
	glCallList( stone_pillar_list );
	glPopMatrix();

	glPushMatrix();
	glTranslatef( -1.3, 0.0, -1.9 );
	glRotatef( -40.0, 0.0, 0.0, -1.0 );
	glCallList( stone_pillar_list );
	glPopMatrix();

	glutSwapBuffers();
}

/******************************************************************************
 *                            keyListener                                     *
 ******************************************************************************/
void keyListener( unsigned char key, int x, int y ) {
	switch (key) {
		case 'q':
		case 27:
			exit(0);
	}
}

/******************************************************************************
 *                       functionKeyListener                                  *
 ******************************************************************************/
void functionKeyListener( int key, int x, int y ) {
	GLfloat xz_radius;

	switch (key) {
		case GLUT_KEY_LEFT :
				camera_hrotation += ROTATION_STEP;
				if ( camera_hrotation > 360.0 ) camera_hrotation -= 360.0;
			break;
		case GLUT_KEY_RIGHT :
				camera_hrotation -= ROTATION_STEP;
				if ( camera_hrotation < 0.0 ) camera_hrotation += 360.0;
			break;
		case GLUT_KEY_UP :
				camera_vrotation += ROTATION_STEP;
				if ( camera_vrotation > 90.0 ) camera_vrotation = 90.0;
			break;
		case GLUT_KEY_DOWN :
				camera_vrotation -= ROTATION_STEP;
				if ( camera_vrotation < 0.0 ) camera_vrotation = 0.0;
			break;
	}
	xz_radius = cos( camera_vrotation * RAD_FACTOR ) * CAMERA_RADIUS;
	camera_position.x = sin( camera_hrotation * RAD_FACTOR ) * xz_radius;
	camera_position.y = sin( camera_vrotation * RAD_FACTOR ) * CAMERA_RADIUS + Y_SOCKET;
	camera_position.z = cos( camera_hrotation * RAD_FACTOR ) * xz_radius;
	setCamera();
	glutPostRedisplay();
}

/******************************************************************************
 *                         menuListener                                       *
 ******************************************************************************/
void menuListener( int selection ) {
	switch ( selection ) {
	case MENU_DAYLIGHT:
		glColorMaterial( GL_FRONT, GL_SPECULAR );
		glLightfv( GL_LIGHT0, GL_DIFFUSE, light_daylight );
		break;
	case MENU_SUNSET:
		glColorMaterial( GL_FRONT, GL_AMBIENT_AND_DIFFUSE );
		glLightfv( GL_LIGHT0, GL_DIFFUSE, light_sunset );
		break;
	case MENU_QUIT:
		exit (0);
	}
	glutPostRedisplay();
}

/******************************************************************************
 *                           cleanup                                          *
 ******************************************************************************/
void cleanup( void ) {
	printf( "Cleaning up ... " );
	discardTextures();
	printf( "done.\n" );
}


/******************************************************************************
 *                           MAIN                                             *
 ******************************************************************************/
int main( int argc, char *argv[] ) {
	printf( "Initializing ... " );
	/* init */
	glutInit( &argc, argv );
	glutInitDisplayMode( GLUT_DOUBLE | GLUT_RGBA | GLUT_DEPTH );
	glutInitWindowPosition( 100, 100 );
	glutInitWindowSize( viewport_width, viewport_height );
	atexit( cleanup );

	/* create window */
	windowHandle = glutCreateWindow( "OpenGL Demo - Martin Christian" );

	printf( "textures, " );
	/* load textures */
	initTextures( textureFiles, TEXTURE_COUNT );

	/* init light, material, GL-states */
	printf( "scene, " );
	initScene();

	/* create menu */
	printf( "menu, " );
	glutCreateMenu( menuListener );
	glutAddMenuEntry( "Daylight", MENU_DAYLIGHT );
	glutAddMenuEntry( "Sunset", MENU_SUNSET );
	glutAddMenuEntry( "Quit", MENU_QUIT );
	glutAttachMenu( GLUT_RIGHT_BUTTON );

	/* register callback functions */
	printf( "callback handler" );
	glutReshapeFunc( resizeWindow );
	glutDisplayFunc( renderScene );
	glutKeyboardFunc( keyListener );
	glutSpecialFunc( functionKeyListener );
	printf( " ... done.\n" );

	/* run */
	glutMainLoop();

	return 0;
}
