#include "l4.h"
#include "assert.h"

#include "linux/mm.h"
#include "asm/page.h"
#include "asm/io.h"
#include "asm/pgalloc.h"
#include "asm/tlbflush.h"

//#define DEBUG_LOOKUP_PTABS
	
pte_t * 
lookup_pte(pgd_t *page_dir, unsigned long pf_address)
{
	/*
	 * find the page table entry within the page table hierarchy
	 */
	pte_t *pte = NULL;
	pgd_t *pgd = page_dir + pgd_index(pf_address);

#ifdef DEBUG_LOOKUP_PTABS
	if ((int)page_dir < 0x1000) {
		printk("%s: page_dir=%x\n", __func__, (int)page_dir);
		enter_kdebug("page_dir<4096");
	}
	printk("%s: %lx pdir = %p", __func__, pf_address, pgd);
#endif
	if (pgd_present(*pgd)) {
		pmd_t *pmd = pmd_offset(pgd, pf_address);
#ifdef DEBUG_LOOKUP_PTABS
		printk(" pmd = %p", pmd);
#endif
		if (pmd_present(*pmd))
			pte = pte_offset_map(pmd, pf_address);
	}
#ifdef DEBUG_LOOKUP_PTABS
	printk(" pte = %p\n", pte);
#endif
	return pte;
}

static inline unsigned long
ptes_same_pmd(pte_t *a, pte_t *b)
{
	unsigned long a_addr = (unsigned long)a;
	unsigned long b_addr = (unsigned long)b;

	return ((a_addr & PAGE_MASK) == (b_addr & PAGE_MASK));
}

/* A semi-optimized get next pte */
static inline pte_t * 
pte_next(pgd_t *page_dir, unsigned long pf_address, pte_t *last)
{
	pte_t *pte = last + 1;

	if ((!last) || !ptes_same_pmd(pte, last))
	{
		pte = lookup_pte(page_dir, pf_address);
	}
#ifdef DEBUG_LOOKUP_PTABS
	else
		printk("%s pte = %p\n", __func__, pte);
#endif

	return pte;
}


/* Flush a page from the user's address space */
void flush_tlb_page(struct vm_area_struct *vma, unsigned long address)
{
	address &= PAGE_MASK;
	flush_tlb_range(vma, address, address + PAGE_SIZE);
}

/* Flush a range of memory from the kernel's virtual address space */
void flush_tlb_kernel_range(unsigned long start, unsigned long end)
{
#if 0
	unsigned long base, count;
	L4_Fpage_t fpage;

	count = 0;
	base = start & PAGE_MASK;

	while (1) {
		fpage = L4_FpageLog2(base, PAGE_SHIFT);

		L4_Set_Rights(&fpage, L4_FullyAccessible);  /* To unmap */
		L4_LoadMR(count++, fpage.raw);

		if (count == __L4_NUM_MRS)
		{
			L4_Unmap(count-1);
			count = 0;
		}

		base += PAGE_SIZE;
		if (base >= end)
		{
			if (count)
				L4_Unmap(count-1);
			return;
		}
	}
#endif
}

/* Flush a single tlb entry in the kernel */
void __flush_tlb_one(unsigned long addr)
{
	addr &= PAGE_MASK;
	flush_tlb_kernel_range(addr, addr + PAGE_SIZE);
}

/* Flush a range of tlb entries in the user's address space
 * in the address range given.
 * This could be optimized to factorize the area into 2^n
 * fpages to reduce the number of upmaps, but the algorithm
 * looks to be too complicated and may cause slowdown.
 */
void flush_tlb_range(struct vm_area_struct *vma, unsigned long start, 
		     unsigned long end)
{
#if 0
	unsigned long base, phys, count;
	L4_Fpage_t fpage;
	pte_t *ptep;

	assert(vma->vm_mm != NULL);

	count = 0;
	base = start & PAGE_MASK;

	ptep = lookup_pte((pgd_t *)vma->vm_mm->pgd, base);

	while (1) {
		if ((ptep != NULL) && pte_mapped(*ptep)) {
			phys = (unsigned long)__va(
					(unsigned long)pte_pfn(*ptep) << PAGE_SHIFT);
			fpage = L4_FpageLog2(phys, PAGE_SHIFT);

			L4_Set_Rights(&fpage, L4_FullyAccessible);  /* To unmap */
			L4_LoadMR(count++, fpage.raw);

			if (count == __L4_NUM_MRS)
			{
				L4_Unmap(count-1);
				count = 0;
			}

			*ptep = pte_mkunmapped(*ptep);
		}

		base += PAGE_SIZE;
		if (base >= end)
		{
			if (count)
				L4_Unmap(count-1);
			return;
		}

		ptep = pte_next((pgd_t *)vma->vm_mm->pgd, base, ptep);
	}
#endif
}

/* This should only be called from set_pte */
void flush_tlb_pte(pte_t *ptep)
{
	unsigned long phys;
	L4_Fpage_t fpage;

	phys = (unsigned long)__va(
			(unsigned long)pte_pfn(*ptep) << PAGE_SHIFT);
	fpage = L4_FpageLog2(phys, PAGE_SHIFT);
	L4_Set_Rights(&fpage, L4_FullyAccessible);  /* To unmap */
	L4_LoadMR(0, fpage.raw);
	L4_Unmap(0);
	// Don't do this as we are only called in set_pte and will be overwritten
	// *ptep = pte_mkunmapped(*ptep);
}

void remove_tlb_pte(pte_t *ptep)
{
	if (pte_mapped(*ptep)) {
		flush_tlb_pte(ptep);
		*ptep = pte_mkunmapped(*ptep);
	}
}

void flush_tlb_mm(struct mm_struct *mm)
{
#if 0
    	L4_Fpage_t fpage;

    	fpage = L4_CompleteAddressSpace;
    	L4_Set_Rights(&fpage, L4_FullyAccessible);  /* To unmap */

	L4_LoadMR(0, fpage.raw);
	L4_Unmap(0);
#endif
}

/* Update the user's address space with the new mapping
 * This does not do the L4 map, but the message is loaded
 * and the reply in the syscall loop handles this.
 */
void
update_mmu_cache(struct vm_area_struct *vma, 
		 unsigned long address, 
		 pte_t pte)
{
	L4_MapItem_t map;
	unsigned long phys;
	pte_t *ptep;

	if (waiting_fault(current->thread_info))
	{
		phys = pte_pfn(pte) << PAGE_SHIFT;

		map = L4_MapItem ( (L4_Fpage_t)
				(L4_FpageLog2 ((unsigned long) phys, PAGE_SHIFT).raw + 
				pte_access(pte)),
				address);
		L4_MsgPut (&current_regs()->msg, 0, 0, (L4_Word_t *) 0, 2, &map);

		ptep = lookup_pte((pgd_t *)vma->vm_mm->pgd, address);
		*ptep = pte_mkmapped(*ptep);

		/* Flush page from cache */
		// XXX - only do this if ptep was mapped before  mkmapped!!
		L4_Set_PageAttribute(L4_FpageLog2 (address, PAGE_SHIFT), L4_FlushCache);
	}
}

