//#include "l4e.h"
#include "assert.h"

#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/time.h>
#include <linux/interrupt.h>
#include <linux/rcupdate.h>
#include <linux/module.h>

#include <iguana/thread.h>
#include <l4/thread.h>

u64 jiffies_64 = INITIAL_JIFFIES;

extern unsigned long wall_jiffies;

extern void interrupt_loop(void);
void show_state(void);

unsigned long long sched_clock(void)
{
	return (unsigned long long)jiffies * (1000000000 / HZ);
}

extern L4_ThreadId_t timer_thread;
extern void (*late_time_init)(void);
static void __init timer_start (void);

void __init
time_init (void)
{
	late_time_init = timer_start;
}

static void __init
timer_start (void) 
{ 
	/* Send message to irq thread -- wake it up */
	L4_Msg_t        msg;
	L4_MsgTag_t     tag;

	L4_MsgClear(&msg);
	L4_MsgLoad(&msg);

	tag = L4_Send(timer_thread);

	assert (!L4_IpcFailed(tag));
	return;
} 

/*
 * This version of gettimeofday has microsecond resolution and better than
 * microsecond precision on fast machines with cycle counter.
 */
void do_gettimeofday(struct timeval *tv)
{
	unsigned long seq;
	unsigned long usec, sec;

	do {
		seq = read_seqbegin(&xtime_lock);
		usec = 0;//do_gettimeoffset();
		{
			unsigned long lost = jiffies - wall_jiffies;
			if (lost)
				usec += lost * (1000000 / HZ);
		}
		sec = xtime.tv_sec;
		usec += (xtime.tv_nsec / 1000);
	} while (read_seqretry(&xtime_lock, seq));

	while (usec >= 1000000) {
		usec -= 1000000;
		sec++;
	}

	tv->tv_sec = sec;
	tv->tv_usec = usec;
}

int do_settimeofday(struct timespec *tv)
{
	time_t wtm_sec, sec = tv->tv_sec;
	long wtm_nsec, nsec = tv->tv_nsec;

	if ((unsigned long)tv->tv_nsec >= NSEC_PER_SEC)
		return -EINVAL;

	write_seqlock_irq(&xtime_lock);
	/*
	 * This is revolting. We need to set "xtime" correctly. However, the
	 * value in this location is the value at the most recent update of
	 * wall time.  Discover what correction gettimeofday() would have
	 * made, and then undo it!
	 */
	nsec -= 0;//do_gettimeoffset() * NSEC_PER_USEC;
	nsec -= (jiffies - wall_jiffies) * TICK_NSEC;

	wtm_sec  = wall_to_monotonic.tv_sec + (xtime.tv_sec - sec);
	wtm_nsec = wall_to_monotonic.tv_nsec + (xtime.tv_nsec - nsec);

	set_normalized_timespec(&xtime, sec, nsec);
	set_normalized_timespec(&wall_to_monotonic, wtm_sec, wtm_nsec);

	time_adjust = 0;		/* stop active adjtime() */
	time_status |= STA_UNSYNC;
	time_maxerror = NTP_PHASE_LIMIT;
	time_esterror = NTP_PHASE_LIMIT;
	write_sequnlock_irq(&xtime_lock);

	return 0;
}

