#include <linux/config.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/err.h>
#include <linux/fs.h>
#include <linux/module.h>
#include <l4.h>

#include <l4/schedule.h>

#include <iguana/thread.h>
#include <iguana/eas.h>

#include "assert.h"

#include <asm/arch.h>
#include <asm/current.h>
#include <asm/page.h>
#include <asm/tlbflush.h>
#include <asm/signal_l4.h>

#include <iguana/eas.h>
#include <iguana/thread.h>

extern void NORET_TYPE syscall_loop (void);
static void new_thread_handler(struct thread_info *prev);
static void new_process_handler(struct thread_info *prev);
extern void schedule_tail(struct task_struct *prev);

extern L4_Fpage_t kip_area, utcb_area;
extern L4_ThreadId_t timer_thread;

#include <l4/ipc.h>
#include <l4/message.h>

long
kernel_thread(int (*fn)(void *), void *arg, unsigned long flags)
{
	long pid;
	//printk("kernel_thread for: %p\n", fn);
	current_thread_info()->request.u.thread.proc = fn;
	current_thread_info()->request.u.thread.arg = arg;
	current_thread_info()->request.op = OP_KTHREAD;
	pid = do_fork(CLONE_VM | flags, 0, NULL, 0, NULL, NULL);
	//printk("kernel_thread for: %p is %ld\n", fn, pid);
	return pid;
}


long
sys_fork(struct pt_regs * regs)
{
	current_thread_info()->request.op = OP_FORK;

	return do_fork(SIGCHLD, ARCH_sp(regs), regs, 0, NULL, NULL);
}

/*
 * This is trivial, and on the face of it looks like it
 * could equally well be done in user mode.
 *
 * Not so, for quite unobvious reasons - register pressure.
 * In user mode vfork() cannot have a stack frame, and if
 * done by calling the "clone()" system call directly, you
 * do not have enough call-clobbered registers to hold all
 * the information you need.
 */
long
sys_vfork(struct pt_regs * regs)
{
	current_thread_info()->request.op = OP_FORK;

	return do_fork(CLONE_VFORK | CLONE_VM | SIGCHLD, ARCH_sp(regs), regs, 0, NULL, NULL);
}


long
sys_execve(char *file, char **argv, char **env)
{
	int error;
	char *filename;
	filename = getname((char *) file);
	error = PTR_ERR(filename);
	if (IS_ERR(filename)) goto out;
	error = do_execve(filename, argv, env, NULL);
	putname(filename);
 out:
	return(error);
}

/* This is the execve entry point for use when called
   from init at kernel startup. At this case at the end
   of it we want to go into the syscall_loop
*/
int
__execve(char *file, char **argv, char **env)
{
	int ret;

	ret = sys_execve(file, argv, env);
	if (ret < 0)
		return ret;

	set_usermode_status_true(current_thread_info());

	/* Reset the stack pointer - forget old context */
	arch_change_stack((void*) (THREAD_SIZE +
				(unsigned long) current_thread_info() -
				sizeof(long)));

	syscall_loop();
	/* Don't return */
	while (1);
}

void
cpu_idle (void) 
{ 
	int r;

	/* Idle thread has special user_tid so we can identify it */
	current_thread_info()->user_tid = L4_anythread;

	//printk("Starting idle loop (%lx)\n", L4_Myself().raw);

	r  = L4_Set_Priority(L4_Myself(), 99);
	if (r == 0) {
		printk("Failed: %lx\n", L4_ErrorCode());
	}
	assert(r != 0);

	atomic_inc(&init_mm.mm_count);
	current->mm = &init_mm;
	current->active_mm = &init_mm;

	while (1) {
		while (!need_resched())
		{
			/* Wait on timer_thread */
			if (!need_resched())
				L4_Receive(timer_thread);
			//printk("#");
		}
		//printk("!");
		schedule();
	}
} 

int
copy_thread(int nr, unsigned long clone_flags, unsigned long usp,
	    unsigned long unused,  struct task_struct * p, 
	    struct pt_regs * regs)
{
	int r = 0;
	//printk("%s %d %lx usp=%lx, task=%p, regs=%p\n", __func__, nr, clone_flags, usp, p, regs);

	if ((p->thread_info->request.op != OP_KTHREAD) &&
	    (current_thread_info()->user_tid.raw != L4_nilthread.raw))
	{
		extern char __wombat_user_fork_handler[];
		L4_Word_t fork_start = TASK_SIG_BASE +
			(((L4_Word_t)&__wombat_user_fork_handler) & ~PAGE_MASK);

		p->thread_info->user_tid = 
			eas_create_thread(p->mm->context, 
					  L4_Myself(), L4_Myself(),
					  (void*) L4_Address(utcb_area)); 
		if (p->thread_info->user_tid.raw == 0)
		    return -ENOMEM;

		{
			char name[16];
			snprintf(name, 15, "L_%d", p->pid);
			L4_KDB_SetThreadName(p->thread_info->user_tid, name);
		}

		r  = L4_Set_Priority(p->thread_info->user_tid, 98);
		assert(r != 0);

		L4_Copy_regs (current_thread_info()->user_tid,
				   p->thread_info->user_tid);

		p->thread_info->request.u.fork.user_sp = usp ? usp : ARCH_sp(regs);
		p->thread_info->request.u.fork.user_ip = ARCH_pc(regs);
		p->thread_info->request.u.fork.user_start = fork_start;
	} else {
		p->thread_info->user_tid = L4_nilthread;
	}

	/* Setup kernel ip/sp */
	/* XXX - setup argument to new process handler */
	if (p->thread_info->request.op == OP_KTHREAD) {
		p->thread_info->request.op = OP_RESUME;

		arch_setup_ipsp(p->thread_info, new_thread_handler,
				(THREAD_SIZE +
				 (unsigned long) p->thread_info -
				 sizeof(long)));
	} else {
		p->thread_info->request.op = OP_RESUME;

		arch_setup_ipsp(p->thread_info, new_process_handler,
				(THREAD_SIZE +
				 (unsigned long) p->thread_info -
				 sizeof(long)));
	}
	return 0;
}


void flush_thread (void) 
{ 
	//printk("flush_thread called\n"); 
} 

void
start_thread(struct pt_regs * regs, unsigned long pc, unsigned long sp)
{ 
	L4_ThreadId_t thrd;
	//printk("%s current = %p  pc = %lx, sp = %lx\n", __func__, current, pc, sp);

	if (current_thread_info()->user_tid.raw == L4_nilthread.raw) {
		int r;
		thrd = eas_create_thread( current->mm->context,
				L4_Myself(), L4_Myself(),
				(void*) L4_Address(utcb_area) ); 

		assert(thrd.raw != 0);
		r  = L4_Set_Priority(thrd, 98);
		assert(r != 0);
		{
			char name[16];
			snprintf(name, 15, "L_%d", current->pid);
			L4_KDB_SetThreadName(thrd, name);
		}
		current_thread_info()->user_tid = thrd;
	} else {
		thrd = current_thread_info()->user_tid;
		L4_AbortIpc_and_stop_Thread(thrd);
	}
	regs = current_regs();

	ARCH_put_pc(regs, pc);
	ARCH_put_sp(regs, sp);

//	syscall_exit(1);

	L4_Start_SpIp (thrd, sp, pc);

        /* Setup which messages we will recieve */
	L4_Accept (L4_UntypedWordsAcceptor);

	/* Wait for the first message */
	current_thread_info()->tag =
		L4_Receive(thrd);

	L4_MsgStore (current_thread_info()->tag,
			&current_regs()->msg); /* Get the tag */
} 


void*
__switch_to(struct task_struct *prev, struct task_struct *next)
{
	struct thread_info *prev_info = prev->thread_info;
	struct thread_info *next_info = next->thread_info;
#if 0
	printk("Switch from: %d to %d\n", prev->pid, next->pid);
#endif
	prev_info->request.op = OP_RESUME;

	local_save_flags(prev_info->irq_flags);

	if (next_info->request.op == OP_KTHREAD) {
		printk("KTHREAD\n");
		WARN_ON(1);
	} else if (next_info->request.op == OP_RESUME) {
		/* Current is set here because it is not stack based */
		/* XXX - is the race from here to the switch important? */
		current_tinfo(smp_processor_id()) = (unsigned long)next_info;

		prev_info = arch_switch(prev_info, next_info);
		mb();
	} else {
		printk("OTHER\n");
		WARN_ON(1);
	}

	local_irq_restore(prev_info->irq_flags);

	return prev_info->task;
}

static void
new_thread_handler(struct thread_info *prev)
{
	//printk("%s current = %p  prev = %p\n", __func__, current, prev->task);

	schedule_tail(prev->task);

	current_thread_info()->request.u.thread.proc(
			current_thread_info()->request.u.thread.arg);

	do_exit(0);
}


L4_INLINE void L4_Start_SpIpUser (L4_ThreadId_t t, L4_Word_t sp, L4_Word_t ip, L4_Word_t user)
{
    L4_Word_t dummy;
    L4_ThreadId_t dummy_id;

    (void) L4_ExchangeRegisters (t, L4_ExReg_sp_ip + L4_ExReg_Resume + L4_ExReg_AbortIPC + L4_ExReg_user,
				 sp, ip, 0, user, L4_nilthread,
				 &dummy, &dummy, &dummy, &dummy, &dummy,
				 &dummy_id);
}

static void
new_process_handler(struct thread_info *prev)
{
//	struct pt_regs * regs;
//	printk("%s current = %p  prev = %p\n", __func__, current, prev->task);

	schedule_tail(prev->task);

	set_usermode_status_true(current_thread_info());

//	regs = current_regs();
//	ARCH_put_pc(regs, current_thread_info()->request.u.fork.user_ip);
//	ARCH_put_sp(regs, current_thread_info()->request.u.fork.user_sp);

//	syscall_exit(1);

	L4_Start_SpIpUser (current_thread_info()->user_tid,
			current_thread_info()->request.u.fork.user_sp,
			current_thread_info()->request.u.fork.user_start,
			current_thread_info()->request.u.fork.user_ip);

        /* Setup which messages we will recieve */
	L4_Accept (L4_UntypedWordsAcceptor);

	/* Wait for the first message */
	current_thread_info()->tag =
		L4_Receive(current_thread_info()->user_tid);

	syscall_loop();
	/* Don't return */
	while (1);
}

void show_regs(struct pt_regs * regs)
{
	printk("show_regs() called\n");
}

