#include "l4.h"
#include "l4/misc.h"
#include "assert.h"

#include "linux/mm.h"
#include "linux/module.h"

#include "iguana/memsection.h"

void
l4e_cache_flush(void) {
	L4_Word_t attributes[4] = {0,0,0,0};
	L4_MemoryControl (-2, attributes);
}

void flush_cache_all (void) { l4e_cache_flush(); } 
void flush_cache_mm (void) { l4e_cache_flush(); } 

void flush_cache_page(struct vm_area_struct *vma, unsigned long page)
{
	L4_Word_t ret = L4_Set_PageAttribute( L4_FpageLog2(page, PAGE_SHIFT),
			L4_FlushCache);
	if (ret != 1)
		l4e_cache_flush();
}

void flush_cache_range(struct vm_area_struct *vma, unsigned long start, unsigned long end)
{
	L4_Word_t i, ret = 1;
	for (i = start & PAGE_MASK; i < end; i += PAGE_SIZE)
	{
		ret = L4_Set_PageAttribute(
				L4_FpageLog2(i, PAGE_SHIFT), L4_FlushCache);
		if (ret != 1)
			break;
	}
	if (ret != 1)
		l4e_cache_flush();
}

void flush_dcache_page(struct page *page)
{
	L4_Word_t ret = L4_Set_PageAttribute(
			L4_FpageLog2((L4_Word_t)page_address(page), PAGE_SHIFT),
			L4_FlushDCache);
	if (ret != 1)
		l4e_cache_flush();
}

void flush_icache_page(struct vm_area_struct *vma, struct page *page)
{
	L4_Word_t ret = L4_Set_PageAttribute(
			L4_FpageLog2((L4_Word_t)page_address(page), PAGE_SHIFT),
			L4_FlushICache);
	if (ret != 1)
		l4e_cache_flush();
}

void flush_data_cache_page(unsigned long addr)
{
	L4_Word_t ret = L4_Set_PageAttribute( L4_FpageLog2(addr, PAGE_SHIFT),
			L4_FlushDCache);
	if (ret != 1)
		l4e_cache_flush();
}

void flush_icache_range(unsigned long start, unsigned long end)
{
	L4_Word_t i, ret = 1;
	for (i = start & PAGE_MASK; i < end; i += PAGE_SIZE)
	{
		ret = L4_Set_PageAttribute(
				L4_FpageLog2(i, PAGE_SHIFT), L4_FlushICache);
		if (ret != 1)
			break;
	}
	if (ret != 1)
		l4e_cache_flush();
}

pte_t *lookup_pte(pgd_t *page_dir, unsigned long address);

void flush_cache_vmap(unsigned long start, unsigned long end)
{
	pte_t *pte;
	unsigned long address, phys;

	for (address = start; address < end; address += PAGE_SIZE) {
		pte = lookup_pte(init_mm.pgd, address);
		assert(pte != NULL);
		phys = pte_pfn(*pte) << PAGE_SHIFT;
		memsection_page_map(vmalloc_memsect, L4_Fpage(phys, PAGE_SIZE), L4_Fpage(address, PAGE_SIZE));
	}
	flush_cache_all();
}

void flush_cache_vunmap(unsigned long start, unsigned long end)
{
	unsigned long address;

	for (address = start; address < end; address += PAGE_SIZE)
		memsection_page_unmap(vmalloc_memsect, L4_Fpage(address, PAGE_SIZE));
	flush_cache_all();
}
