#include "l4.h"
#include "assert.h"

#include <linux/config.h>

#include <linux/init.h>
#include <linux/console.h>
#include <linux/spinlock.h>
#include <linux/jiffies.h>
#include <linux/serial.h>

#include <linux/tty.h>
#include <linux/tty_driver.h>
#include <linux/tty_flip.h>

#include <linux/mm.h>

#include <asm/uaccess.h>
#include <serial_client.h>
#include <iguana/thread.h>
#include <iguana/object.h>
#include <iguana/session.h>
#include <iguana/cap.h>
#include <iguana/memsection.h>
#include <iguana/pd.h>
#include <iguana/debug.h>
#include <interfaces/iguana_client.h>

static struct tty_driver *l4e_serial_driver;
extern void *console_out, *console_in;
object_t *output;

extern uintptr_t temp_cap_slot;
extern uintptr_t temp_cap_used;
extern uintptr_t temp_cap_size;
extern uintptr_t temp_cap_addr;

struct tty_struct *the_tty; /* Global -- hack!*/

static uintptr_t
_stream_write(object_t *stream_obj, const uintptr_t length, 
	      const uintptr_t data, CORBA_Environment *env)
{
	/* Find the session */
	return stream_write(stream_obj->server, stream_obj->obj, 
			    length, data, env);
}

static void l4e_console_write(struct console *co, const char *p,
			       unsigned count)
{
	char c[16];
	unsigned i, n;
	int donecr = 0;

	if (output == NULL) {
		static object_t obj_interface;
		session_ref_t session;
		thread_ref_t serv_ref;
		uintptr_t ignore;
		memsection_ref_t memsection;
		output = &obj_interface;

		obj_interface.obj = (objref_t) console_out;
		//session = session_create_full_share(obj_interface.obj, &obj_interface.server);

		memsection = memsection_lookup(obj_interface.obj, &serv_ref);
		obj_interface.server = thread_l4tid(serv_ref);

		/* 
		   We can't use library functions yet, so we do this the old
		   fashioned way 
		*/
		session = iguana_pd_create_session(L4_Pager(), 
					 pd_myself(), thread_myself(), serv_ref, 
					 memsection_lookup((uintptr_t)iguana_get_clist(0), &ignore), 
					 NULL).ref.session;
		assert(session != 0);
	}

	i = n = 0;
	while (count > 0 || i > 0) {
		if (count > 0 && i < sizeof(c)) {
			if (p[n] == '\n' && !donecr) {
				c[i++] = '\r';
				donecr = 1;
			} else {
				c[i++] = p[n++];
				donecr = 0;
				--count;
			}
		} else {
			_stream_write(output, i, (uintptr_t) c, NULL);
			i = 0;
		}
	}

}

static struct tty_driver *l4e_console_device(struct console *c, int *index)
{

	*index = c->index;
	return l4e_serial_driver;
}

static struct console l4econs = {
	.name	    = "ttyS",
	.write	    = l4e_console_write,
	.device	    = l4e_console_device,
	.flags	    = CON_PRINTBUFFER,
	.index	    = -1,
};

static int __init l4e_console_init(void)
{
	register_console(&l4econs);
	return 0;
}

console_initcall(l4e_console_init);

/* 
 * The TTY driver
 *	from:
 *	linux/arch/alpha/kernel/srmcons.c
 */
static spinlock_t l4e_serial_callback_lock = SPIN_LOCK_UNLOCKED;

#define MAX_L4E_CONSOLE_DEVICES 1	/* only support 1 console device */

struct l4e_serial_private {
	struct tty_struct *tty;
	spinlock_t lock;
};

/* called with callback_lock held */
void
l4e_serial_receive_chars(char *addr, int length)
{
	if (! the_tty) {
		return;
	}
	while(length-- > 0) {
		if (*addr == 0x0b)	// CTRL-K
			L4_KDB_Enter("debug");
		if (*addr == 0x14)	// CTRL-T
			debug_dump_info();
		else
			tty_insert_flip_char(the_tty, *addr, 0);
		addr++;
	}
	tty_schedule_flip(the_tty);
}

/* called with callback_lock held */
static int
l4e_serial_do_write(struct tty_struct *tty, const unsigned char *buf, int count)
{
	long c, remaining = count;
	unsigned char *cur;

	for (cur = (unsigned char *)buf; remaining > 0; ) {
		/* 
		 * Break it up into reasonable size chunks to allow a chance
		 * for input to get in
		 */
		c = min_t(long, 128L, remaining);
		{
			/* FIXME: we assume that console_write has setup 
			   the stream tid */
			_stream_write(output, c, (uintptr_t) cur, NULL);
		}

		remaining -= c;
		cur += c;
	}
	return count;
}

static int
l4e_serial_write(struct tty_struct *tty, int from_user,
	      const unsigned char *buf, int count)
{
	unsigned long flags;
	if (from_user) {
		unsigned char tmp[512];
		int ret = 0;
		size_t c;

		while ((c = count) > 0) {
			if (c > sizeof(tmp))
				c = sizeof(tmp);
			
			c -= copy_from_user(tmp, buf, c);

			if (!c) { 
				printk("%s: EFAULT (count %d)\n",
				       __FUNCTION__, count);
				return -EFAULT;
			}

			spin_lock_irqsave(&l4e_serial_callback_lock, flags);
			l4e_serial_do_write(tty, tmp, c);
			spin_unlock_irqrestore(&l4e_serial_callback_lock, flags);

			buf += c;
			count -= c;
			ret += c;
		}

		return ret;
	}

	spin_lock_irqsave(&l4e_serial_callback_lock, flags);
	l4e_serial_do_write(tty, buf, count);
	spin_unlock_irqrestore(&l4e_serial_callback_lock, flags);

	return count;
}

static int 
l4e_getserial(struct tty_struct *tty, struct serial_struct *retinfo)
{
	struct serial_struct tmp;

	memset(&tmp, 0, sizeof(tmp));

	if (copy_to_user(retinfo, &tmp, sizeof(*retinfo)))
		return -EFAULT;
        return 0;
}

static int
l4e_serial_ioctl(struct tty_struct *tty, struct file *filp, unsigned int cmd,
		unsigned long arg)
{
	int ret = -ENOIOCTLCMD;

	switch (cmd) {
	case TIOCGSERIAL:
		if ((ret = verify_area(VERIFY_WRITE, (void *) arg,
					sizeof(struct serial_struct))) == 0)
                        ret = l4e_getserial(tty, (struct serial_struct *) arg);
		break;

	case TIOCSSERIAL:
		ret = 0;
		break;

	case TIOCSERGWILD: /* obsolete */
	case TIOCSERSWILD: /* obsolete */
		ret = 0;
		break;
	}
	return ret;
}

static int
l4e_serial_write_room(struct tty_struct *tty)
{
	return 512;
}

static int
l4e_serial_chars_in_buffer(struct tty_struct *tty)
{
	return 0;
}

static int
l4e_serial_get_private_struct(struct l4e_serial_private **ps)
{
	static struct l4e_serial_private *l4e_p = NULL;
	static spinlock_t l4e_p_lock = SPIN_LOCK_UNLOCKED;
	unsigned long flags;
	int retval = 0;

	spin_lock_irqsave(&l4e_p_lock, flags);

	do {
		if (l4e_p != NULL) {
			*ps = l4e_p;
			break;
		}

		l4e_p = kmalloc(sizeof(*l4e_p), GFP_KERNEL);
		if (l4e_p == NULL) {
			retval = -ENOMEM;
			break;
		}

		l4e_p->tty = NULL;
		l4e_p->lock = SPIN_LOCK_UNLOCKED;

		*ps = l4e_p;
	} while(0);

	spin_unlock_irqrestore(&l4e_p_lock, flags);

	return retval;
}

static int
l4e_serial_open(struct tty_struct *tty, struct file *filp)
{
	struct l4e_serial_private *l4e_p;
	unsigned long flags;
	int retval;

	retval = l4e_serial_get_private_struct(&l4e_p);
	if (retval)
		return retval;

	spin_lock_irqsave(&l4e_p->lock, flags);

	if (!l4e_p->tty) {
		tty->driver_data = l4e_p;
		the_tty = l4e_p->tty = tty;
	}

	spin_unlock_irqrestore(&l4e_p->lock, flags);

	return 0;
}

static void
l4e_serial_close(struct tty_struct *tty, struct file *filp)
{
	struct l4e_serial_private *l4e_p = tty->driver_data;
	unsigned long flags;

	spin_lock_irqsave(&l4e_p->lock, flags);

	if (tty->count == 1) {

		l4e_p->tty = NULL;
	}

	spin_unlock_irqrestore(&l4e_p->lock, flags);
}


static struct tty_operations l4e_serial_ops = {
	.open		= l4e_serial_open,
	.close		= l4e_serial_close,
	.write		= l4e_serial_write,
	.write_room	= l4e_serial_write_room,
	.chars_in_buffer= l4e_serial_chars_in_buffer,
	.ioctl		= l4e_serial_ioctl,
};


static int __init
l4e_serial_init(void)
{
	struct tty_driver *driver;
	int err;

	driver = alloc_tty_driver(MAX_L4E_CONSOLE_DEVICES);
	if (!driver)
		return -ENOMEM;
	driver->driver_name = "l4e";
	driver->name = "ttyS";
	driver->major = TTY_MAJOR;
	driver->minor_start = 64;
	driver->type = TTY_DRIVER_TYPE_SERIAL;
	driver->subtype = SERIAL_TYPE_NORMAL;
	driver->init_termios = tty_std_termios;
	driver->init_termios.c_cflag =
		B9600 | CS8 | CREAD | HUPCL | CLOCAL;
	driver->flags = TTY_DRIVER_REAL_RAW;
	/* Setup read call */

	tty_set_operations(driver, &l4e_serial_ops);
	err = tty_register_driver(driver);
	if (err) {
		put_tty_driver(driver);
		return err;
	}
	l4e_serial_driver = driver;

	return -ENODEV;
}

late_initcall(l4e_serial_init);

