/****************************************************************************
 *
 * Copyright (C) 2007, Leipzig University
 *
 * File path:	src/platform/hellas/intctrl.cc
 * Description:	Interrupt controler.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Author: Martin Christian, Leipzig University
 ***************************************************************************/

#include <macros.h>
#include INC_GLUE(intctrl.h)

/* Interrupt Controller class */
intctrl_t intctrl;

/* Interrupt callback handler */
void handle_interrupt(word_t irq);

/*************************** Initalization methods **************************/

SECTION(".init") void intctrl_t::init_arch()
{
	mpsc_ch0 = (struct mpsc_uart*)MPSC_CHANNEL0;
	mpsc0cr = (u32_t*)MPSC_CAUSE0;
	mpsc0mr = (u32_t*)MPSC_MASK0;
	main_int_cause_l = (u32_t*)MAIN_CAUSE_L;
	main_int_cause_h = (u32_t*)MAIN_CAUSE_H;
	cpu_int_mask_l = (u32_t*)CPU_MASK_L;
	*cpu_int_mask_l = 0;
	cpu_int_mask_h = (u32_t*)CPU_MASK_H;
	*cpu_int_mask_h = 0;

	/* Enable debug port for I/O */
	mpsc_ch0->chr4 = (u32_t)(MPSC_CHR4_Z);
	mpsc_ch0->chr5 = (u32_t)(MPSC_CHR5_INT | MPSC_CHR5_V);
	mpsc_ch0->chr2 = (u32_t)(MPSC_CHR2_EHUNT);

	return;
}

SECTION(".init") void intctrl_t::init_cpu( word_t cpu )
{
	*mpsc0mr = 0;
	*mpsc0cr = 0;
	*cpu_int_mask_h |= (u32_t)MAIN_INT_MPSC0;
	return;
}

/**************************** Information methods ***************************/

word_t intctrl_t::get_number_irqs()
{
	return IRQS;
}

bool intctrl_t::is_irq_available( word_t irq )
{
	return ( (irq <= 0) && (irq < IRQS) );
}

/************************************ Misc **********************************/

void intctrl_t::set_cpu( word_t irq, word_t cpu )
{
	switch ( irq ) {
		case MPSC_INT_RCC:
			TRACE( "Set CPU on MPSC_INT_RCC\n" );
			*mpsc0mr |= (u32_t)MPSC_CMR0_RCC;
			break;
		default:
			TRACE( "Set CPU on unknown interrupt\n" );
			break;
	}
}

/**
 * Act on interrupt
 */
void intctrl_t::handle_irq( word_t irq )
{
	/* Search for interrupt cause */
	if ( *main_int_cause_h & MAIN_INT_MPSC0 ) { // is MPSC0 interrupt?
		if ( *mpsc0cr & MPSC_CMR0_RCC ) { // is RCC interrupt?
			irq = MPSC_INT_RCC;
		}
		else {
			TRACE( "Unknown interrupt on MPSC0\n" );
			return;
		}
	}
	else {
		TRACE( "Unknown interrupt on Marvell Bridge\n" );
		return;
	}

	/* Handle interrupt */
	this->mask_and_ack( irq );
	handle_interrupt( irq );
}
