/****************************************************************************
 *
 * Copyright (C) 2007, Leipzig University
 *
 * File path:	platform/hellas/io.cc
 * Description:	Raw i/o wrappers.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Author: Martin Christian, Leipzig University
 ***************************************************************************/

#include <macros.h>
#include <debug.h>
#include <types.h>
#include <kdb/console.h>
#include INC_ARCH(string.h)
#include INC_PLAT(marvell.h)

#define CONSOLE_NAME "mpsc0"

/* Use channel 0 to send and receive debugging information. */
static volatile struct mpsc_uart *mpsc_ch0 = (struct mpsc_uart*)MPSC_CHANNEL0;

/**
 * Assume that interrupts are enabled in CHR5.INT and the RCC bit in MPSC mask
 * is set. Then the RCC bit in MPSC cause is set if a character is received.
 * mpsc0cr = MPSC Channel 0 Cause Register
 * mpsc0mr = MPSC Channel 0 Mask Register
 */
static volatile u32_t *mpsc0cr = (u32_t*)MPSC_CAUSE0;
static volatile u32_t *mpsc0mr = (u32_t*)MPSC_MASK0;

/**
 * Activate MPSC debug console
 */
static void switch_console( const char *name ) {
	for( int i = 0; kdb_consoles[i].name; i++ )
		if( !strcmp(kdb_consoles[i].name, name) ) {
			kdb_current_console = i;
			return;
		}
}

/**
 * Initialize MPSC UART debug input and output channel.
 * Since initialization for debug output was already done at
 * bootup, we only need to set the bits necessary to receive
 * characters.
 */
void mpsc_init_console( void )
{
	/* Set BCE=0x0 and enable debug port Z=1 */
	mpsc_ch0->chr4 = (u32_t)(MPSC_CHR4_Z);
	/* Set CNTL[0]=0x0 which enables debug input */
	mpsc_ch0->chr5 = (u32_t)(MPSC_CHR5_INT | MPSC_CHR5_V);
	/* Enable hunting */
	mpsc_ch0->chr2 = (u32_t)(MPSC_CHR2_EHUNT);
	asm volatile( "eieio ;" );

	switch_console( CONSOLE_NAME );
	printf( "Activated debug console on MPSC channel 0.\n" );
}

/*
 * Write character to console. 
 */
static void mpsc_putc( char c )
{
	if ( c == '\n' ) {
		mpsc_ch0->chr1 = MPSC_CHR1_TCS((u32_t)'\r');
		mpsc_ch0->chr2 = (u32_t)MPSC_CHR2_TTCS;
		while ( (mpsc_ch0->chr2) & (u32_t)MPSC_CHR2_TTCS )
			; /* wait until character is sent */
	}
	mpsc_ch0->chr1 = MPSC_CHR1_TCS((u32_t)c);
	mpsc_ch0->chr2 = (u32_t)MPSC_CHR2_TTCS;
	asm volatile( "eieio ;" );
	while ( (mpsc_ch0->chr2) & ((u32_t)MPSC_CHR2_TTCS) )
		; /* wait until character is sent */
}

/*
 * Read character from console.
 */
static char mpsc_getc( bool block )
{
	u32_t tmp;
	u32_t old_mask;
	char c = 0;

	/* Save mask and enable RCC interrupt */
	old_mask = *mpsc0mr;
	*mpsc0mr = (u32_t)MPSC_CMR0_RCC;
	asm volatile( "eieio ;" );

	while ( !((*mpsc0cr) & (u32_t)MPSC_CMR0_RCC) && (block) )
		; /* wait until character is available */
	if ( (*mpsc0cr) & (u32_t)MPSC_CMR0_RCC ) {
		tmp = mpsc_ch0->chr10;
		/* extract character */
		c = (char)((tmp >> 8) & (u32_t)0x000000ff);
		/* write character back to clear rcrn */
		mpsc_ch0->chr10 = tmp;
		/* clear interrupt flag */
		*mpsc0cr &= 0xbfffffff;
	}

	/* Restore mask */
	*mpsc0mr = old_mask;
	asm volatile( "eieio ;" );
	return c;
}

/****************************************************************************
 *    Console registration
 ****************************************************************************/

/*
 * kernel/include/kdb/console.h:
 * 
 * class kdb_console_t {
 * public:
 *   char * name;
 *   void (*init) (void);
 *   void (*putc) (char c);
 *   char (*getc) (bool block);
 * };
 */

kdb_console_t kdb_consoles[] = {
	{ CONSOLE_NAME, mpsc_init_console, mpsc_putc, mpsc_getc },
	KDB_NULL_CONSOLE
};
word_t kdb_current_console = 0;
