/****************************************************************************
 *
 * Copyright (C) 2007, Leipzig University
 *
 * File path:	platform/hellas/intctrl.h
 * Description:	Marvell interrupt controller.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Author: Martin Christian, Leipzig University
 ***************************************************************************/

#ifndef __PLATFORM__HELLAS__INTCTRL_H__
#define __PLATFORM__HELLAS__INTCTRL_H__

#include <debug.h> // includes macros TRACE, TRACE_INIT, TRACEF
#include INC_PLAT(marvell.h)

#define IRQS	1

/**
 * Interrupt enumeration
 */
enum {
	MPSC_INT_RCC = 0
};

/*
 * Interrupt controler.
 */
class intctrl_t {
	private:
		/* MPSC Channel 0 to send and receive debugging information. */
		volatile struct mpsc_uart *mpsc_ch0;
		/* MPSC Channel 0 Cause Register */
		volatile u32_t *mpsc0cr;
		/* MPSC Channel 0 Mask Register */
		volatile u32_t *mpsc0mr;
		/* Main Interrupt Cause Register Low */
		volatile u32_t *main_int_cause_l;
		/* Main Interrupt Cause Register High */
		volatile u32_t *main_int_cause_h;
		/* CPU Interrupt Mask Register Low */
		volatile u32_t *cpu_int_mask_l;
		/* CPU Interrupt Mask Register High */
		volatile u32_t *cpu_int_mask_h;


	public:
		/* L4 init methods */
		void init_arch();
		void init_cpu( word_t cpu );
		/* Interrupt controlling methods */
		void mask( word_t irq );
		bool unmask( word_t irq );
		void ack( word_t irq );
		void mask_and_ack( word_t irq );
		void enable( word_t irq );
		void disable( word_t irq );

		/* set affinity/routing */
		void set_cpu( word_t irq, word_t cpu );

		/* Misc methods */
		word_t get_number_irqs();
		bool is_irq_available( word_t irq );

		/* handler invoked on interrupt */
		void handle_irq( word_t irq );

	/* Endianess I/O methods
	private:
		void out32le( word_t reg, u32_t val );
		void out32be( word_t reg, u32_t val );
		u32_t in32le( word_t reg );
		u32_t in32be( word_t reg );
	*/
};

/*********************** Inline methods stay in header **********************/

/**
 * The interrupt number irq is related to the enum entry in marvell.h
 */
INLINE void intctrl_t::enable( word_t irq )
{
	switch ( irq ) {
		case MPSC_INT_RCC:
			TRACE( "Enable MPSC_INT_RCC\n" );
			*mpsc0mr |= (u32_t)MPSC_CMR0_RCC;
			break;
		default:
			break;
	}
	return;
}

/**
 * The interrupt number irq is related to the enum entry in marvell.h
 */
INLINE void intctrl_t::disable( word_t irq )
{
	switch ( irq ) {
		case MPSC_INT_RCC:
			TRACE( "Disable MPSC_INT_RCC\n" );
			*mpsc0mr &= ~(u32_t)MPSC_CMR0_RCC;
			break;
		default:
			break;
	}
	return;
}

/**
 * mask = disable
 */
INLINE void intctrl_t::mask( word_t irq )
{
	disable( irq );
	return;
}

/**
 * unmask = enable, but unmask returns true if an IRQ was already pending
 */
INLINE bool intctrl_t::unmask( word_t irq )
{
	enable( irq );
	switch ( irq ) {
		case MPSC_INT_RCC:
			return ( (*mpsc0cr) & (u32_t)MPSC_CMR0_RCC );
			break;
	}
	return false;
}

/**
 * 
 */
INLINE void intctrl_t::ack( word_t irq )
{
	switch ( irq ) {
		case MPSC_INT_RCC:
			TRACE( "ACK MPSC_INT_RCC\n" );
			*mpsc0cr &= ~(u32_t)MPSC_CMR0_RCC;
			break;
		default:
			break;
	}
	return;
}

/**
 * 
 */
INLINE void intctrl_t::mask_and_ack( word_t irq )
{
	this->mask( irq );
	this->ack( irq );
	return;
}

#endif	/* __PLATFORM__HELLAS__INTCTRL_H__ */
