/*********************************************************************
 *                
 * Copyright (C) 2002, 2003-2004,  Karlsruhe University
 * Copyright (C) 2007, Leipzig University
 *                
 * File path:     glue/v4-powerpc/tcb.h
 * Description:   specific implementations
 *                
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *                
 * Authors: Someone, Karlsruhe University
 *          Martin Christian, Leipzig University
 ********************************************************************/
#ifndef __GLUE__V4_POWERPC__TCB_H__
#define __GLUE__V4_POWERPC__TCB_H__

#include INC_ARCH(ppc_registers.h)
#include INC_ARCH(msr.h)
#include INC_API(syscalls.h)
#include INC_GLUE(resources_inline.h)

#define TRACE_TCB(x...)
//#define TRACE_TCB(x...)	TRACEF(x)

/********************************************************************** 
 *
 *                      processor state
 *
 **********************************************************************/

/* NOTE: The size must be a multiple of 8-bytes.
 * Important: the ordering here is rather important.  It matches hard coded
 * offsets in the inlined assembler thread switch code, offsets in the
 * threadswitch fast path, and offsets in the notify inlined assembler.
 */
typedef struct {
    word_t ip;
    word_t unused;
    word_t r31;
    word_t r30;
} tswitch_frame_t;

/* NOTE: The size must be a multiple of 8-bytes.
 * Important: ordering for back_chain and lr_save is important, as they
 * must sit on the stack in positions defined by the eabi.
 */
typedef struct {
    word_t back_chain;	// eabi prior stack location
    word_t lr_save;	// eabi return address
    word_t unused;
    word_t arg2;
    word_t arg1;
    void (*func)( word_t, word_t );
} notify_frame_t;


INLINE addr_t get_kthread_ip( tcb_t *tcb )
{
    tswitch_frame_t *tswitch_frame = (tswitch_frame_t *)tcb->stack;
    return (addr_t)tswitch_frame->ip;
}

INLINE syscall_regs_t *get_user_syscall_regs( tcb_t *tcb )
{
    return (syscall_regs_t *)
	(word_t(tcb->get_stack_top()) - sizeof(syscall_regs_t));
}

INLINE except_regs_t *get_user_except_regs( tcb_t *tcb )
{
    return (except_regs_t *)
	(word_t(tcb->get_stack_top()) - sizeof(except_regs_t));
}

/********************************************************************** 
 *
 *                      tcb methods
 *
 **********************************************************************/

INLINE void tcb_t::set_utcb_location( word_t location )
{
    utcb_t *dummy = (utcb_t *)NULL;
    utcb_location = location + (word_t)dummy->mr;
}

INLINE word_t tcb_t::get_utcb_location()
{
    utcb_t *dummy = (utcb_t *)NULL;
    return utcb_location - (word_t)dummy->mr;
}

INLINE void tcb_t::set_global_id(threadid_t tid)
{
    this->myself_global = tid;
    get_utcb()->my_global_id = tid;
}

INLINE void tcb_t::set_cpu( cpuid_t cpu )
{
#if defined(CONFIG_SMP)
	this->cpu = cpu;
	get_utcb()->processor_no = cpu;
#endif
}

/**
 * tcb_t::get_mr: returns value of message register
 * @index: number of message register
 */
INLINE word_t tcb_t::get_mr(word_t index)
{
    return get_utcb()->mr[index];
}

/**
 * tcb_t::set_mr: sets the value of a message register
 * @index: number of message register
 * @value: value to set
 */
INLINE void tcb_t::set_mr(word_t index, word_t value)
{
    get_utcb()->mr[index] = value;
}

/**
 * copies a set of message registers from one UTCB to another
 * @param dest destination TCB
 * @param start MR start index
 * @param count number of MRs to be copied
 */
INLINE void tcb_t::copy_mrs(tcb_t * dest, word_t start, word_t count)
{
	ASSERT( ALWAYS, start + count <= IPC_NUM_MR );
	ASSERT( ALWAYS, count > 0 );
	start--;
	while ( count ) {
		dest->get_utcb()->mr[start+count] = this->get_utcb()->mr[start+count];
		count--;
	}
}

/**
 * Called if r flag in ExchangeRegisters control is set.
 * Helps OS-Layer to implement fork() semantics:
 * Copies all user registers except SP, UTCB and IP
 * from src stack frame to this stack frame.
 */
INLINE void tcb_t::copy_user_regs( tcb_t *src )
{
	except_regs_t *to = get_user_except_regs( this );
	except_regs_t *from = get_user_except_regs( src );

	to->xer = from->xer;
	to->cr = from->cr;
	to->ctr = from->ctr;
	to->r0 = from->r0;
	to->r3 = from->r3;
	to->r4 = from->r4;
	to->r5 = from->r5;
	to->r6 = from->r6;
	to->r7 = from->r7;
	to->r8 = from->r8;
	to->r9 = from->r9;
	to->r10 = from->r10;
	to->r11 = from->r11;
	to->r12 = from->r12;
	to->r13 = from->r13;
	to->r14 = from->r14;
	to->r15 = from->r15;
	to->r16 = from->r16;
	to->r17 = from->r17;
	to->r18 = from->r18;
	to->r19 = from->r19;
	to->r20 = from->r20;
	to->r21 = from->r21;
	to->r22 = from->r22;
	to->r23 = from->r23;
	to->r24 = from->r24;
	to->r25 = from->r25;
	to->r26 = from->r26;
	to->r27 = from->r27;
	to->r28 = from->r28;
	to->r29 = from->r29;
	to->r30 = from->r30;
	to->r31 = from->r31;
	to->lr = from->lr;
	to->srr1_flags = from->srr1_flags;
}

/**
 * If bit 12 (RegsToMRs) is set in the ExchangeRegisters control
 * field, the following registers are copied into message registers:
 * r0     -> MR[0],
 * IP     -> MR[1],
 * MSR    -> MR[2],
 * r3-r31 -> MR[3-31]
 *
 * The RegsToMRs flag is not defined in N1Rev5 API.
 * 
 * TODO: There are only 32 message registers in the UTCB but 38
 * register values to be saved. The IP is returned here, because
 * it's needed for a fork() service. Stack and UTCB are separate
 * for each thread, so they are not needed for a fork().
 */
INLINE void tcb_t::copy_regs_to_mrs(tcb_t *src)
{
	except_regs_t *from = get_user_except_regs( src );

	set_mr( 0, from->r0 );
	set_mr( 1, from->srr0_ip );
	set_mr( 2, from->srr1_flags );
	set_mr( 3, from->r3 );
	set_mr( 4, from->r4 );
	set_mr( 5, from->r5 );
	set_mr( 6, from->r6 );
	set_mr( 7, from->r7 );
	set_mr( 8, from->r8 );
	set_mr( 9, from->r9 );
	set_mr( 10, from->r10 );
	set_mr( 11, from->r11 );
	set_mr( 12, from->r12 );
	set_mr( 13, from->r13 );
	set_mr( 14, from->r14 );
	set_mr( 15, from->r15 );
	set_mr( 16, from->r16 );
	set_mr( 17, from->r17 );
	set_mr( 18, from->r18 );
	set_mr( 19, from->r19 );
	set_mr( 20, from->r20 );
	set_mr( 21, from->r21 );
	set_mr( 22, from->r22 );
	set_mr( 23, from->r23 );
	set_mr( 24, from->r24 );
	set_mr( 25, from->r25 );
	set_mr( 26, from->r26 );
	set_mr( 27, from->r27 );
	set_mr( 28, from->r28 );
	set_mr( 29, from->r29 );
	set_mr( 30, from->r30 );
	set_mr( 31, from->r31 );
}

INLINE bool tcb_t::allocate()
{
    // Write to the tcb, to ensure that the kernel maps this tcb
    // with write access.  Write to the bottom of the stack.
    // TODO: should we do this?  It wastes a cache line.

    // (word_t *)( (word_t)this + sizeof(tcb_t) ) = 0;
    get_kernel_space()->allocate_tcb(this);
    return true;
}

INLINE void tcb_t::set_space(space_t * space)
{
    this->space = space;
    if( EXPECT_FALSE(space == NULL) )
    {
	/* Thread switch expects pdir_cache to be 0 for kernel threads.
	 */
	this->pdir_cache = 0;
	this->resources.set_kernel_thread( this );
	return;
    }

    this->pdir_cache = (word_t)space->get_segment_id().raw;
    TRACE_TCB("set_space(), space 0x%p, tcb 0x%p, kernel_space 0x%p\n", 
	      space, this, get_kernel_space() );

    space->sync_kernel_space( this );	/* Map this tcb into the space. */
    space->handle_hash_miss( this );	/* Install this tcb into the pg hash. */

    space->handle_hash_miss( space );	/* TODO: is this the solution? */
}

INLINE word_t * tcb_t::get_stack_top()
{
    word_t stack;
    /* The powerpc eabi stack must be 8-byte aligned. */
    stack = ((word_t)this + TOTAL_TCB_SIZE) & ~(8-1);
    return (word_t *)stack;
}

INLINE void tcb_t::init_stack()
{
    this->stack = get_stack_top();
    TRACE_TCB( "stack = %p, tcb bottom = %p, tcb size = %d\n", 
	       this->stack, this, sizeof(tcb_t) );
}

/**
 * read value of the acceptor
 */
INLINE acceptor_t tcb_t::get_acceptor(void)
{
    return get_utcb()->acceptor;
}

/**
 * set the value of the acceptor register
 * @param value value to set
 */
INLINE void tcb_t::set_acceptor(const acceptor_t value)
{
    get_utcb()->acceptor = value;
}


/********************************************************************** 
 *
 *                      thread switch routines
 *
 **********************************************************************/

/**
 * tcb_t::switch_to: switches to specified tcb
 */
INLINE void tcb_t::switch_to(tcb_t * dest)
{
    ASSERT( ALWAYS, dest->stack );
    ASSERT( ALWAYS, get_cpu() == dest->get_cpu() );
    ASSERT( ALWAYS, dest != this );

    // TODO: adjust the thread switch return address to load 
    // resources.  Thus the common path need not check for a load.
    if( EXPECT_FALSE(this->resource_bits) )
	this->resources.save( this );

    /* NOTE: pdir_cache holds the segment ID. */

    register word_t dummy0 asm("r25");
    register word_t dummy1 asm("r26");
    register word_t dummy2 asm("r27");
    register word_t dummy3 asm("r28");
    register word_t dummy4 asm("r29");

    asm volatile (
	    "stw %%r30, -4(%%r1) ;"	/* Preserve r30 on the stack. */
	    "stw %%r31, -8(%%r1) ;"	/* Preserve r31 on the stack. */
	    "mtsprg " MKSTR(SPRG_CURRENT_TCB) ", %3 ;"	/* Save the tcb pointer in sprg1. */
	    "lis %3, 1f@ha ;"		/* Grab the return address. */
	    "la  %3, 1f@l(%3) ;"	/* Put the return address in %3. */
	    "stwu %3, -16(%%r1) ;"	/* Store (with update) the return address on the
					   current stack. */
	    "stw %%r1, 0(%2) ;"		/* Save the current stack in
					   *old_stack. */

	    "cmplw cr0, %0, %4 ;"	/* Compare the new seg ID to the old. */
	    "cmplwi cr1, %0, 0 ;"	/* Is kernel thread? */
	    "cror eq, cr0*4+eq, cr1*4+eq ;"	/* OR the results. */
	    "beq 0f ;"			/* Skip addr space switch if possible. */
	    "isync ;"		// TODO: does the kernel access user space?
#if defined(CONFIG_PPC_SEGMENT_LOOP)
	    /* Here is a loop to set the segment registers.  It is 4 cycles
	     * slower than the nonlooped version.  But it has 17 fewer
	     * instructions totalling 68 bytes, and thus saves 
	     * over 2 cache lines.
	     */
	    "li %%r3, 12 ;"	// Init the loop count.
	    "mtctr %%r3 ;"	// Load the loop count.
	    "li %%r3, 0 ;"	// Init the segment register index.
	    "99:" 
	    "mtsrin %0, %%r3 ;"	// Set the segment register.
	    "addi %0, %0, 1 ;"	// Increment the segment ID.
	    "extlwi %%r3, %0, 4, 28; "	// Extract the segment register index.
	    "bdnz 99b ;"	// Loop.
#else
	    "mtsr 0, %0 ; addi %0, %0, 1 ;"
	    "mtsr 1, %0 ; addi %0, %0, 1 ;"
	    "mtsr 2, %0 ; addi %0, %0, 1 ;"
	    "mtsr 3, %0 ; addi %0, %0, 1 ;"
	    "mtsr 4, %0 ; addi %0, %0, 1 ;"
	    "mtsr 5, %0 ; addi %0, %0, 1 ;"
	    "mtsr 6, %0 ; addi %0, %0, 1 ;"
	    "mtsr 7, %0 ; addi %0, %0, 1 ;"
	    "mtsr 8, %0 ; addi %0, %0, 1 ;"
	    "mtsr 9, %0 ; addi %0, %0, 1 ;"
	    "mtsr 10, %0 ; addi %0, %0, 1 ;"
	    "mtsr 11, %0 ;"
#endif
	    "isync ;"		// TODO: can we instead rely on rfi ?

	    "0:"
	    "addi %%r1, %1, 16 ;"	/* Install the new stack. */
	    "lwz %%r3, -16(%%r1) ;"	/* Grab the new thread's address. */
	    "lwz %%r30, -4(%%r1) ;"	/* Restore r30. */
	    "lwz %%r31, -8(%%r1) ;"	/* Restore r31. */
	    "mtctr %%r3 ;"		/* Prepare to jump. */
	    "bctr ;"			/* Jump to the new thread. */

	    "1:"			/* The return address. */

	    : "=r" (dummy0), "=r" (dummy1), "=r" (dummy2), "=r" (dummy3),
	      "=r" (dummy4)
	    : "0" (dest->pdir_cache), "1" (dest->stack), "2" (&this->stack), 
	      "3" (dest), "4" (this->pdir_cache)
	    : "memory", "r5", "r6", "r7", "r8", "r9", 
	      "r10", "r11", "r12", "r13", "r14", "r15", "r16", "r17", "r18",
	      "r19", "r20", "r21", "r22", "r23", "r24",
	      "ctr", "lr", 
	      "cr0", "cr1", "cr2", "cr3", "cr4", "cr5", "cr6", "cr7"
#if (__GNUC__ >= 3)
	      , "xer"
#endif
	    );

    if( EXPECT_FALSE(this->resource_bits) )
	this->resources.load( this );
}

/**********************************************************************
 *
 *                        notification functions
 *
 **********************************************************************/
INLINE void tcb_t::notify( void (*func)() )
{
    this->notify( (void (*)(word_t, word_t))func, 0, 0 );
}

INLINE void tcb_t::notify( void (*func)(word_t), word_t arg1 )
{
    this->notify( (void (*)(word_t, word_t))func, arg1, 0 );
}

/* 
 * access functions for ex-regs'able registers
 */
INLINE addr_t tcb_t::get_user_ip()
{
    return addr_t(get_user_syscall_regs(this)->srr0_ip);
}

INLINE addr_t tcb_t::get_user_sp()
{
    return addr_t(get_user_syscall_regs(this)->r1_stack);
}

INLINE word_t tcb_t::get_user_flags()
{
    return get_user_syscall_regs(this)->srr1_flags & MSR_USER_MASK;
}

INLINE void tcb_t::set_user_ip(addr_t ip)
{
    get_user_syscall_regs(this)->srr0_ip = word_t(ip);
}

INLINE void tcb_t::set_user_sp(addr_t sp)
{
    get_user_syscall_regs(this)->r1_stack = word_t(sp);
}

INLINE void tcb_t::set_user_flags(const word_t flags)
{
    get_user_syscall_regs(this)->srr1_flags = 
	(flags & MSR_USER_MASK) | MSR_USER;
}

INLINE void tcb_t::return_from_ipc (void)
{
    return_ipc_abort();
}

INLINE void tcb_t::return_from_user_interruption (void)
{
    word_t return_stack;
    extern word_t _except_return_shortcircuit[];

    // We want to short-circuit the return trip, to the exception
    // exit path.  So we jump to the point in assembler code which
    // starts restoring the user's full exception context.

    return_stack = (word_t)this->get_stack_top() - 
	(sizeof(except_regs_t) + EABI_STACK_SIZE);

    // Install the stack, and jump to the context store code.
    asm volatile (
	    "mtlr %0 ;"
	    "mr %%r1, %1 ;"
	    "blr ;"
	    :
	    : "r" ((word_t)_except_return_shortcircuit), "r" (return_stack)
	    : "r0"
	    );
}

/**********************************************************************
 *
 *                        in-kernel IPC invocation
 *
 **********************************************************************/

/**
 * tcb_t::do_ipc: invokes an in-kernel IPC
 * @param to_tid destination thread id
 * @param from_tid from specifier
 * @return IPC message tag (MR0)
 */
INLINE msg_tag_t tcb_t::do_ipc( threadid_t to_tid, threadid_t from_tid )
{
    this->resources.set_kernel_ipc( this );

#if defined(CONFIG_SYSV_ABI)
    register word_t r3 asm("r3") = (word_t)&to_tid;
    register word_t r4 asm("r4") = (word_t)&from_tid;
#else
    register word_t r3 asm("r3") = to_tid.get_raw();
    register word_t r4 asm("r4") = from_tid.get_raw();
#endif

    /* ABI stack */
    asm volatile (
	    "addi %%r1, %%r1, -16 ;"	/* Allocate stack space for r30 and r31,
					   and the ABI stack space for calling
					   a function. */
	    "stw %%r30, 8(%%r1) ;"	/* Preserve r30. */
	    "stw %%r31, 12(%%r1) ;"	/* Preserve r31. */
	    "bl sys_ipc ;"		/* Call sys_ipc(). */
	    "lwz %%r31, 12(%%r1) ;"	/* Restore r31. */
	    "lwz %%r30, 8(%%r1) ;"	/* Restore r30. */
	    "addi %%r1, %%r1, 16 ;"	/* Clean-up the stack. */
	    : "+r" (r3), "+r" (r4)
	    : 
	    : "r0", "r2", "r5", "r6", "r7", "r8", "r9", "r10", "r11", "r12", "r13", 
	      "r14", "r15", "r16", "r17", "r18", "r19", "r20", "r21", 
	      "r22", "r23", "r24", "r25", "r26", "r27", "r28", "r29", 
	      "ctr", "lr", "cr0", "cr1", "cr2", "cr3", "cr4", "cr5", 
	      "cr6", "cr7", "memory"
#if (__GNUC__ >= 3)
	      , "xer"
#endif
	    );

    this->resources.clr_kernel_ipc( this );

    msg_tag_t tag;
    tag.raw = this->get_mr(0);
    return tag;
}

/**********************************************************************
 *
 *                       preemption callback signaling
 *
 **********************************************************************/

/**
 * set the address where preemption occured
 */
INLINE void tcb_t::set_preempted_ip(addr_t ip)
{
    get_utcb()->preempted_ip = (word_t)ip;
}

INLINE addr_t tcb_t::get_preempted_ip()
{
    return (addr_t)get_utcb()->preempted_ip;
}

/**
 * get the preemption callback ip
 */
INLINE addr_t tcb_t::get_preempt_callback_ip()
{
    return (addr_t)get_utcb()->preempt_callback_ip;
}


/**********************************************************************
 *
 *                        global tcb functions
 *
 **********************************************************************/

__attribute__ ((const)) INLINE tcb_t * addr_to_tcb (addr_t addr)
{
    return (tcb_t *) ((word_t) addr & KTCB_MASK);
}

INLINE void set_sprg_tcb( tcb_t *tcb )
{
    ppc_set_sprg( SPRG_CURRENT_TCB, (word_t)tcb );
}

__attribute__ ((const)) INLINE tcb_t *get_sprg_tcb()
{
    return (tcb_t *)ppc_get_sprg( SPRG_CURRENT_TCB );
}

__attribute__ ((const)) INLINE tcb_t * get_current_tcb()
{
    return addr_to_tcb( __builtin_frame_address(0) );
}

#if defined(CONFIG_SMP)
INLINE cpuid_t get_current_cpu()
{
    return get_idle_tcb()->get_cpu();
}
#endif

/**
 * initial_switch_to: switch to first thread
 * @param tcb TCB of initial thread.
 *
 * Switches to the initial thread.  The stack is expected to contain a
 * notify frame.
 * We use this function, rather than tcb_t::switch_to(), because the outgoing
 * stack isn't a valid tcb.
 */
INLINE void NORETURN initial_switch_to( tcb_t *tcb )
{
    // Store the target thread's tcb in the appropriate sprg.
    set_sprg_tcb( tcb );

    // Activate the thread switch frame.
    asm volatile (
	    "mtctr %0 ;"	// Prepare to branch.
	    "mr %%r1, %1 ;"	// Install the new stack.
	    "bctr ;"		// Branch to the instruction pointer.
	    : /* outputs */
	    : /* inputs */
	      "r" (get_kthread_ip(tcb)), "b" (tcb->stack)
	    );

    while( 1 );
}

/**
 * adds a thread to the space
 * @param tcb pointer to thread control block
 */
INLINE void space_t::add_tcb(tcb_t * tcb)
{
    /* Make sure the valid bit does not get set.
     * NB. this is a subtree entry */
    x.thread_count += 2;
#ifdef CONFIG_DEBUG
    spaces_list_lock.lock();
    ENQUEUE_LIST_TAIL(x.thread_list, tcb, thread_list);
    spaces_list_lock.unlock();
#endif
}

/**
 * removes a thread from a space
 * @param tcb_t thread control block
 * @return true if it was the last thread
 */
INLINE bool space_t::remove_tcb(tcb_t * tcb)
{
    ASSERT( ALWAYS, x.thread_count != 0 );
    /* Make sure the valid bit does not get set */
    x.thread_count -= 2;
#ifdef CONFIG_DEBUG
    spaces_list_lock.lock();
    DEQUEUE_LIST(x.thread_list, tcb, thread_list);
    spaces_list_lock.unlock();
#endif
    return (x.thread_count == 0);
}

#endif /* __GLUE__V4_POWERPC__TCB_H__ */

