/*********************************************************************
 *                
 * Copyright (C) 2002-2004,  Karlsruhe University
 *                
 * File path:     arch/ia64/pib.h
 * Description:   Processor Interrupt Block
 *                
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *                
 * $Id: pib.h,v 1.4 2004/04/22 17:42:54 skoglund Exp $
 *                
 ********************************************************************/
#ifndef __ARCH__IA64__PIB_H__
#define __ARCH__IA64__PIB_H__


/**
 * Default location of the Processor Interrupt Block.
 */
#define IA64_DEFAULT_PIB_ADDR	(0xfee00000UL)


class xtp_t
{
    union {
	struct {
	    u8_t priority	: 4;
	    u8_t		: 3;
	    u8_t disable	: 1;
	};
	u8_t raw;
    };
};


/**
 * An Interrupt Processor Block.  The ib_base specifies the base
 * address of the block.
 */
template<u64_t ib_base> class pib_t
{
public:

    enum type_e {
	INT	= 0,
	PMI	= 2,
	NMI	= 4,
	INIT	= 5,
	ExtINT	= 7
    };

    /**
     * Send IPI to indicated CPU.
     * @param id_eid	id/eid style id for destination CPU
     * @param vector	vector to invoke IPI on
     * @param redirect	allow IPI to be redirected by platform
     * @param type	type of interrupt to generate on destination CPU
     */
    void ipi (u16_t id_eid, u8_t vector,
	      bool redirect = false, type_e type = INT)
	{
	    ((volatile word_t *) ib_base)
		[((word_t) id_eid << 1) + (redirect ? 1 : 0)] = 
		(word_t) vector + ((word_t) type << 7);
	}

    /**
     * Send IPI to indicated CPU.
     * @param id	id for destination CPU  (as taken from in cr.lid)
     * @param eid	eid for destination CPU (as taken from in cr.lid)
     * @param vector	vector to invoke IPI on
     * @param redirect	allow IPI to be redirected by platform
     * @param type	type of interrupt to generate on destination CPU
     */
    void ipi (u8_t id, u8_t eid, u8_t vector,
	      bool redirect = false, type_e type = INT)
	{
	    ipi (((u16_t) id << 8) + eid, vector, redirect, type);
	}

    /**
     * Set the external task priority (XTP) for current CPU.
     * @param xtp	new xtp value
     */
    void set_xtp (xtp_t xtp)
	{
	    ((volatile u8_t *) ib_base)[0x1e0008] = xtp.raw;
	}

    /**
     * Retrieve the interrupt vector when interrupt is generated by an
     * 8529A-compatible interrupt controlled.
     * @return interrupt vector
     */
    u8_t get_inta (void)
	{
	    return ((volatile u8_t *) ib_base)[0x1e0000];
	}
};


#endif /* !__ARCH__IA64__PIB_H__ */
