/*********************************************************************
 *                
 * Copyright (C) 2007,  Leipzig University
 *                
 * File path:     libs/drv_powerpc_timer/src/main.c
 * Description:   Timer based on PowerPC decremeter
 *                
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *                
 * Author: Martin Christian, Leipzig University
 ********************************************************************/
#include <driver.h>
#include <driver/timer_ops.h>
#include <stddef.h>
#include <stdio.h>
#include <inttypes.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <l4/types.h>
#include <l4/ipc.h>
#include <l4/kdebug.h>
#include <iguana/thread.h> /* thread_create_priority */

/* Timer specific section */

#include "powerpc.h"

/* Private functions */
static inline uint64_t get_timebase(void);

/* Device Ops */
static void *setup(int spacec, bus_space_t *spacev, dma_handle_t dma, bus_space_t pciconf);
static void  enable(void *device);
static void  cleanup(void *device);
static void  interrupt(void *device);
/* Timer Ops */
static int timeout(void *device, uint64_t time, callback_t callback, 
		   callback_data_t callback_data, uintptr_t key);
static uint64_t current_time(void *device);

/* Thread data */
#define STACK_SIZE 200
L4_ThreadId_t main_tid, timer_tid;
uintptr_t timer_stack[STACK_SIZE];

/* Driver data */
struct powerpc_dev {
	struct driver_instance generic;
	int enabled;
};

static struct timer_ops ops = {
	/* Driver ops */
	{ setup,
	  enable,
	  cleanup,
	  interrupt },
	/* Timer ops */
	timeout,
	current_time
};

struct driver drv_powerpc_timer = 
{
	.name = "PowerPC timer",
	.class = timer_device_e,
	.ops.t_ops = &ops
};

/* Timer values */
callback_t timer_callback;
callback_data_t timer_data;
uintptr_t timer_key;
static int in_interrupt = 0;
uint64_t next_timeout = -1;

/* Private functions */
static inline uint64_t get_timebase() {
	uint64_t timebase;

	asm volatile (
		"mr    %%r4, %0 ;"
		"mftbu %%r5 ;"
		"mftbl %%r6 ;"
		"stw   %%r5, 0(%%r4) ;"
		"stw   %%r6, 4(%%r4) ;"
		: /* outputs */
		: /* inputs */
		"r" (&timebase)
		: /* clobbers */
		"r4", "r5", "r6", "memory"
	);

	return timebase;
}

/* Device Ops */
static void* setup( int spacec, bus_space_t *spacev,
                    dma_handle_t dma, bus_space_t pciconf )
{
	struct powerpc_dev *drv;
	drv = malloc( sizeof(struct powerpc_dev) );
	if (drv == 0) return 0;
	memset( drv, 0, sizeof(*drv) );
	drv->generic.classp = &drv_powerpc_timer;

	main_tid = L4_Myself();
	//thread_create_priority( 200, &timer_tid );
	thread_create( &timer_tid );
	L4_KDB_SetThreadName( timer_tid, "tmr_thrd" );
	printf( "timer setup: myself = %p, tmr_thrd = %p\n", main_tid.raw, timer_tid.raw );

	return drv;
}

static void enable(void *device) {}

static void cleanup(void *device) {}

static void interrupt( void *device ) {
	in_interrupt = 1;
	timer_callback( timer_data, 0, NULL, timer_key );
	in_interrupt = 0;
}

/* Timer Ops */
static void timer_thread( void ) {
	L4_Msg_t msg;
	int64_t current;

	while(1) {
		if ( next_timeout != -1 ) {
			/* Converstion to ticks from us */
			next_timeout = US_TO_TICKS(next_timeout);
			current = get_timebase();
			while ( current < next_timeout ) {  /* wait */
				current = get_timebase();
			}
		} else {
			/* No timeout, so sleep forever */
			/* Hack: L4_WaitForever() returns for some reason.
			 *       Thus, I'll use a loop instead.
			 */
			// L4_WaitForever();
			while ( 1 ) current++;
		}
		next_timeout = -1;
		L4_MsgClear( &msg );
		L4_Set_Label( &msg.tag, IRQ_LABEL );
		L4_MsgLoad( &msg );
		L4_Call( main_tid );
	}
	assert(!"Shouldn't get here\n");
}

static int timeout( void *device,
                    uint64_t time,
                    callback_t callback,
                    callback_data_t callback_data,
                    uintptr_t key )
{
	L4_Word_t dummy;
	L4_ThreadId_t dummy_id;
	next_timeout = time;
	timer_callback = callback;
	timer_data = callback_data;
	timer_key = key;

	/* Exreg */
	if (in_interrupt == 0) {
		L4_ExchangeRegisters(
		  timer_tid,
		  L4_ExReg_sp_ip | L4_ExReg_Resume | L4_ExReg_AbortIPC | L4_ExReg_pager,
		  (uintptr_t)&timer_stack[STACK_SIZE-8], (uintptr_t)timer_thread, 
		  0, 0, L4_Pager(), &dummy, &dummy, &dummy, &dummy, &dummy, &dummy_id );
	}
	return 0;
}

static uint64_t current_time( void *device )
{
	uint64_t time = get_timebase();
	return TICKS_TO_US(time);
}
